package com.mattsmeets.macrokey.handler.hook;

import com.mattsmeets.macrokey.config.ModConfig;
import com.mattsmeets.macrokey.config.ModState;
import com.mattsmeets.macrokey.event.ExecuteOnTickEvent;
import com.mattsmeets.macrokey.model.LayerInterface;
import com.mattsmeets.macrokey.model.lambda.ExecuteOnTickInterface;

import net.minecraft.client.gui.screen.IngameMenuScreen;
import net.minecraft.client.gui.screen.Screen;
import net.minecraft.client.gui.widget.button.Button;
import net.minecraft.client.resources.I18n;
import net.minecraft.util.text.StringTextComponent;
import net.minecraft.util.text.TranslationTextComponent;
import net.minecraftforge.client.event.GuiScreenEvent;
import net.minecraftforge.common.MinecraftForge;
import net.minecraftforge.eventbus.api.SubscribeEvent;

/**
 * Add the switch layer button to the main menu.
 */
public class GuiEventHandler {
    private Button switchButton = null;
    private ModState modState;

    public GuiEventHandler(final ModState modState) {
        this.modState = modState;
    }

    //----------
    // Event handlers
    //----------

    /**
     * Create the switch layer button.
     *
     * @param event The init GUI event.
     */
    @SubscribeEvent
    public void init(final GuiScreenEvent.InitGuiEvent event) {
        final Screen gui = event.getGui();

        if (isNotMainMenu(event.getGui())) return;
        if (isSwitchButtonDisabled()) return;

        switchButton = new Button( gui.width / 2 + ModConfig.buttonLayerSwitchSettingsX.get(), gui.height / 4 + ModConfig.buttonLayerSwitchSettingsY.get(),
                ModConfig.buttonLayerSwitchSettingsW.get(), ModConfig.buttonLayerSwitchSettingsH.get(), new StringTextComponent(getLayerButtonLabel(modState.getActiveLayer())),
                b -> {
                	b.setMessage(new StringTextComponent(getLayerButtonLabel(modState.nextLayer())));
                }
        );

        event.addWidget(switchButton);
    }

    /**
     * Handle right click.
     * Open the GUI.
     *
     * @param event The mouse click event.
     */
    @SubscribeEvent(receiveCanceled = true)
    public void mouseClickedEvent(final GuiScreenEvent.MouseClickedEvent.Post event) {
        if (isNotMainMenu(event.getGui())
                || isSwitchButtonDisabled()
                || switchButton == null
                || !switchButton.isHovered()) {
            return;
        }

        MinecraftForge.EVENT_BUS.post(new ExecuteOnTickEvent(ExecuteOnTickInterface.openMacroKeyGUI));
    }

    /**
     * Render the tooltip.
     *
     * @param event The draw screen event.
     */
    @SubscribeEvent(receiveCanceled = true)
    public void render(final GuiScreenEvent.DrawScreenEvent.Post event) {
        if (isNotMainMenu(event.getGui())
                || isSwitchButtonDisabled()
                || switchButton == null
                || !switchButton.isHovered()) {
            return;
        }
        event.getGui().renderTooltip(event.getMatrixStack(),
                new TranslationTextComponent("text.layer.hover.right_click"),
                event.getMouseX(),
                event.getMouseY()
        );
    }

    //----------
    // Helpers
    //----------

    private static boolean isNotMainMenu(Screen gui) {
        return !(gui instanceof IngameMenuScreen);
    }

    private static boolean isSwitchButtonDisabled() {
        return ModConfig.buttonLayerSwitchDisabled.get();
    }

    private static String getLayerButtonLabel(final LayerInterface layer) {
        return I18n.get("text.layer.display",
                layer == null ? I18n.get("text.layer.master") : layer.getDisplayName()
        );
    }
}
